const axios = require('axios');
const bcrypt = require('bcrypt');
const crypto = require("crypto");
const jwt = require('jsonwebtoken');
const config = require('../../configs/jwt.configs.js');
const sendEmail = require("../utils/sendEmail");
const XLSX = require("xlsx");

const { HTTP_SUCCESS_RESPONSE } = require("../utils/success");
const { HTTP400Error, HTTP404Error, catchAsync } = require("../utils/errors");

const handlebars = require("handlebars");
const fs = require("fs");
const path = require("path");
const resetPasswordTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/resetPassword.handlebars"), "utf-8"); 
const verifyEmailTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/verifyEmail.handlebars"), "utf-8"); 

// Import model
const Provider = require('../models/providers.js');
const Order = require('../models/orders.js');

const saveAddressCoordinates = async (address) => {
    return await axios(`https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${process.env.GOOGLE_MAP_API_KEY}`)
    .then((response) => {
        return {status:200, data: response.data.results[0] }
    })
    .catch((error) => {
        console.log(error);
        return {status:400, data: error.response }
    })
}

const removeSpace = (str) => {
    return str.replace(/\s/g, "");
};

module.exports = {

    saveCoordinates : async (req, res) => {
        await axios(`https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${process.env.GOOGLE_MAP_API_KEY}`)
        .then((response) => {
            return res.json(response.data.results[0]);
        })
        .then((error) => {
            return res.json(error.response);
        })
    },

    create: catchAsync(async (req, res) => {
        const providerData = { companyName, mainContactFirstName, mainContactLastName, certifiedMedicalExaminarFirstName, certifiedMedicalExaminarLastName, email, phone, password, physicalAddress, mailingAddress, nrcmeNo, nrcmeExpirationDate, onlineAppointmentCalendarLink, practitionerType, appointmentType, providerType, testPrice } = req.body;
        let hashedPassword = bcrypt.hashSync(password, 8)
        let existProvider = await Provider.find({ email: email});
        if(existProvider.length < 1) {
            let provider = new Provider(providerData);
            provider.password = hashedPassword;
            provider.testPrice = providerType == 'preferred' ? 65: Number(testPrice);
            return provider.save()
            .then(async function(provider) {
                let address = `${provider.physicalAddress.street}, ${provider.physicalAddress.city}, ${provider.physicalAddress.state}, ${provider.physicalAddress.zip}`;
                let coordinatesData = await saveAddressCoordinates(address);
                if(coordinatesData.status == 200) {
                    provider.addressCoordinates ={ latitude: coordinatesData.data.geometry?coordinatesData.data.geometry.location.lat:'', longitude: coordinatesData.data.geometry?coordinatesData.data.geometry.location.lng:'', place_id: coordinatesData.data.place_id?coordinatesData.data.place_id:'' }
                }
                provider.token = crypto.randomBytes(32).toString("hex");

                const template = handlebars.compile(verifyEmailTemplate);
                let message = (template({
                    verifyEmailLink: `${process.env.API_BASE_URL}/api/v1/provider/users/verify/${provider._id}/${provider.token}`,
                }));

                sendEmail(provider.email, "Verify Your Email", message);

                await provider.setCreatedBy(provider._id); 
                await provider.setUpdatedBy(provider._id);
                await provider.save(); 
                return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Provider User Successfully Created.' ); 
            }).catch(err => {
                throw new HTTP400Error(err.message);
            });    
            
        } else {
            throw new HTTP400Error('Eamil Already Exists');
        }
    }),

    createProviderByAdmin: catchAsync(async (req, res) => {
        const providerData = { companyName, mainContactFirstName, mainContactLastName, certifiedMedicalExaminarFirstName, certifiedMedicalExaminarLastName, email, phone, password, physicalAddress, mailingAddress, nrcmeNo, nrcmeExpirationDate, onlineAppointmentCalendarLink, practitionerType, appointmentType, providerType, testPrice } = req.body;
        let hashedPassword = bcrypt.hashSync(password, 8)
        let existProvider = await Provider.find({ email: email});
        if(existProvider.length < 1) {
            let provider = new Provider(providerData);
            provider.password = hashedPassword;
            provider.testPrice = providerType == 'preferred' ? 65: Number(testPrice);
            return provider.save()
            .then(async function(provider) {
                let address = `${provider.physicalAddress.street}, ${provider.physicalAddress.city}, ${provider.physicalAddress.state}, ${provider.physicalAddress.zip}`;
                let coordinatesData = await saveAddressCoordinates(address);
                if(coordinatesData.status == 200) {
                    provider.addressCoordinates ={ latitude: coordinatesData.data.geometry?coordinatesData.data.geometry.location.lat:'', longitude: coordinatesData.data.geometry?coordinatesData.data.geometry.location.lng:'', place_id: coordinatesData.data.place_id?coordinatesData.data.place_id:'' }
                }
                provider.token = crypto.randomBytes(32).toString("hex");

                const template = handlebars.compile(resetPasswordTemplate);
                let message = (template({
                    resetPasswordLink: `${process.env.PROVIDER_PANEL_BASE_URL}/password-reset?id=${provider._id}&&token=${provider.token}`,
                }));

                sendEmail(provider.email, "Reset password", message);
                provider.isApproved = true;
                provider.isVerified = true;
                await provider.setCreatedBy(provider._id); 
                await provider.setUpdatedBy(provider._id);
                await provider.save(); 
                return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Provider user Successfully Created.' ); 
            }).catch(err => {
                throw new HTTP400Error(err.message);
            });    
            
        } else {
            throw new HTTP400Error('Eamil Already Exists');
        }
    }),

    login: catchAsync(async(req, res) => {
        const { email, password } = req.body;
        let provider = await Provider.findOne({ email: email });
        if(!provider ) throw new HTTP404Error('Email Does Not Exist.');
        provider = provider.toJSON(); 
        let passwordIsValid = await bcrypt.compareSync(password, provider.password);
        if (!passwordIsValid) throw new HTTP400Error('Password does not matched.');

        if(!provider.isVerified) throw new HTTP400Error('Your email is not verified. Please verify.');
        if(!provider.isApproved) throw new HTTP400Error('You are not approved by admin. Please contact with support.');

        let token = jwt.sign({ _id: provider._id, email: provider.email }, config.secret, { expiresIn: '60m' });
        let refreshToken = jwt.sign({ _id: provider._id, email: provider.email }, config.secret, { expiresIn: '120m' });
        res.setHeader("Authorization", token);
        return HTTP_SUCCESS_RESPONSE(res, { token: token, refreshToken: refreshToken }, 'Auth Successful' );
    }),

    update: async(req, res) => {
        const { companyName, mainContactFirstName, mainContactLastName, certifiedMedicalExaminarFirstName, certifiedMedicalExaminarLastName, email, phone, physicalAddress, mailingAddress, nrcmeNo, nrcmeExpirationDate, onlineAppointmentCalendarLink, practitionerType, appointmentType, providerType, testPrice } = req.body;
        let id = req.params.id;
        let provider = await Provider.findById(id);
        if(!provider) throw new HTTP404Error('Provider not found');
        try {
            provider.companyName = companyName;
            provider.mainContactFirstName = mainContactFirstName;
            provider.mainContactLastName = mainContactLastName;
            provider.certifiedMedicalExaminarFirstName = certifiedMedicalExaminarFirstName;
            provider.certifiedMedicalExaminarLastName = certifiedMedicalExaminarLastName;
            provider.phone = phone;
            provider.email = email;
            provider.physicalAddress = physicalAddress;
            provider.mailingAddress = mailingAddress;
            provider.nrcmeNo = nrcmeNo;
            provider.nrcmeExpirationDate = nrcmeExpirationDate;
            provider.onlineAppointmentCalendarLink = onlineAppointmentCalendarLink;
            provider.practitionerType = practitionerType;
            provider.appointmentType = appointmentType;
            provider.providerType = providerType;
            provider.testPrice = providerType == 'preferred' ? 65: Number(testPrice);
            provider.appointmentType = appointmentType;
            await provider.setUpdatedBy(req.provider ? req.provider._id : req.admin._id); 
            await provider.setUpdatedAt();
            await provider.save();
            return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Provider successfully updated' );
        } catch (err) {
            throw new HTTP400Error(err.message);
        }
    },

    updatePassword: catchAsync(async(req, res) => {
        const { currentPassword, newPassword, confirmPassword } = req.body;
        let id = req.provider._id;
        let provider = await Provider.findById(id);
        if(!provider) throw new HTTP404Error('Provider not found.');

        let passwordIsValid = await bcrypt.compareSync(currentPassword, provider.password);
        if (!passwordIsValid) throw new HTTP400Error('Current password does not matched.');

        if(newPassword !== confirmPassword) {
            return res.status(400).send({ message: "New password and confirm password does not matched." });
        }

        try {
            let hashedPassword = bcrypt.hashSync(newPassword, 8);
            provider.password = hashedPassword;
            await provider.save();
            return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Password successfully updated.' );
        } catch (err) {
            throw new HTTP400Error(err.message);
        }
    }),

    updatePrice: catchAsync(async(req, res) => {
        const { testPrice } = req.body;
        let id = req.params.id;
        let provider = await Provider.findById(id);
        if(!provider) throw new HTTP404Error('Provider not found');
        provider = provider.toJSON();
        provider.testPrice = testPrice;  
        try {
            let updatedProvider = await Provider(provider);
            await updatedProvider.setUpdatedBy(req.provider._id); 
            await updatedProvider.setUpdatedAt();
            updatedProvider = await Provider(updatedProvider); 
            return HTTP_SUCCESS_RESPONSE(res, { data: updatedProvider }, 'Provider test price successfully updated' );
        } catch(err) {
            throw new HTTP400Error(err.message);
        }
    }),

    list: catchAsync(async(req, res) => {
        let providerList = [];
        const { page = 1, limit = 5 } = req.query;
        countDocuments = 0;
        if(req.query.searchData) {
            providerList = await Provider.find({ companyName: { $regex: req.query.searchData, $options: 'i' }} )
                            .select({ companyName: 1, mainContactFirstName: 1, mainContactLastName: 1, certifiedMedicalExaminarFirstName : 1, certifiedMedicalExaminarLastName: 1, phone: 1, email: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1, nrcmeNo: 1, nrcmeExpirationDate: 1, onlineAppointmentCalendarLink: 1, practitionerType: 1, appointmentType: 1, providerType: 1, isApproved: 1, isVerified: 1, createdAt: 1, addressCoordinates: 1 })
                            .sort({createdAt: -1})
                            .limit(limit * 1)
                            .skip((page - 1) * limit)
                            .exec();
            countDocuments = await Provider.find({ companyName: { $regex: req.query.searchData, $options: 'i' }}).countDocuments();
        } else {
            providerList = await Provider.find({})
                            .select({ companyName: 1, mainContactFirstName: 1, mainContactLastName: 1, certifiedMedicalExaminarFirstName : 1, certifiedMedicalExaminarLastName: 1, phone: 1, email: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1, nrcmeNo: 1, nrcmeExpirationDate: 1, onlineAppointmentCalendarLink: 1, practitionerType: 1, appointmentType: 1, providerType: 1, isApproved: 1, isVerified: 1, createdAt: 1, addressCoordinates: 1 })
                            .sort({createdAt: -1})
                            .limit(limit * 1)
                            .skip((page - 1) * limit)
                            .exec();

            countDocuments = await Provider.find({}).countDocuments();
        }

        let totalPages = Math.ceil(countDocuments / limit);

        return HTTP_SUCCESS_RESPONSE(res, { data: providerList, page, totalPages }, 'Provider list successfully retrieved.' );
    }),

    providerListDriverPanel: catchAsync(async(req, res) => {
        let providerList = [];
        let centreLatitude = 0;
        let centreLongitude = 0;
        let minCurrentLat = 0;
        let maxCurrentLat = 0;
        let minCurrentLng = 0;
        let maxCurrentLng = 0;
        let coordinatesData = {
            status: 400
        };

        if(req.query.cityState) {
            req.query.cityState = removeSpace(req.query.cityState);
        }

        if(req.query.zip) {
            req.query.zip = removeSpace(req.query.zip);
        }

        if(req.query.cityState) {
            coordinatesData = await saveAddressCoordinates(req.query.cityState);
        } else if(req.query.zip) {
            coordinatesData = await saveAddressCoordinates(req.query.zip);
        };

        if(coordinatesData.status == 200) {
            centreLatitude = coordinatesData.data.geometry ? coordinatesData.data.geometry.location.lat : 0;
            centreLongitude = coordinatesData.data.geometry ? coordinatesData.data.geometry.location.lng : 0;
        };

        console.log(centreLatitude, centreLongitude, 'lolo');

        if (centreLatitude != 0) {
            minCurrentLat = Number(centreLatitude) - 0.683579;
            maxCurrentLat = Number(centreLatitude) + 0.683579;
        }

        if (centreLongitude != 0) {
            minCurrentLng = Number(centreLongitude) - 0.73792;
            maxCurrentLng = Number(centreLongitude) + 0.73792;
        }

        console.log(centreLatitude, centreLongitude, maxCurrentLat, minCurrentLat, maxCurrentLng, minCurrentLng, 'Location lat lng');

        if(req.query.cityState && req.query.zip ) {
            if(req.query.cityState.includes(',')) {
                const cityState = req.query.cityState.split(',');
                providerList = await Provider.find({ isVerified: true, isApproved: true, $or: [{'physicalAddress.city': { $regex: cityState[0], $options: 'i' }}, {'physicalAddress.state': { $regex: cityState[1], $options: 'i' }}, { 'physicalAddress.zip': { $regex: req.query.zip, $options: 'i' }}, { $and: [{ $and: [{ "addressCoordinates.latitude" : { $gte : minCurrentLat } }, { "addressCoordinates.latitude" : { $lte : maxCurrentLat } }] }, { $and: [{ "addressCoordinates.longitude" : { $gte : minCurrentLng } }, { "addressCoordinates.longitude" : { $lte : maxCurrentLng } }] }]  } ] })
                .select({ companyName:1, email: 1, addressCoordinates: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1 }).sort({createdAt: -1});
            } else {
                providerList = await Provider.find({ isVerified: true, isApproved: true, $or: [{'physicalAddress.city': { $regex: req.query.cityState, $options: 'i' }}, {'physicalAddress.state': { $regex: req.query.cityState, $options: 'i' }}, { 'physicalAddress.zip': { $regex: req.query.zip, $options: 'i' }}, { $and: [{ $and: [{ "addressCoordinates.latitude" : { $gte : minCurrentLat } }, { "addressCoordinates.latitude" : { $lte : maxCurrentLat } }] }, { $and: [{ "addressCoordinates.longitude" : { $gte : minCurrentLng } }, { "addressCoordinates.longitude" : { $lte : maxCurrentLng } }] }] }] })
                .select({ companyName:1, email: 1, addressCoordinates: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1 }).sort({createdAt: -1});
            }
        } else if(req.query.cityState) {
            if(req.query.cityState.includes(',')) {
                const cityState = req.query.cityState.split(',');
                providerList = await Provider.find({ isVerified: true, isApproved: true, $or: [{'physicalAddress.city': { $regex: cityState[0], $options: 'i' }}, {'physicalAddress.state': { $regex: cityState[1], $options: 'i' }}, { $and: [{ $and: [{ "addressCoordinates.latitude" : { $gte : minCurrentLat } }, { "addressCoordinates.latitude" : { $lte : maxCurrentLat } }] }, { $and: [{ "addressCoordinates.longitude" : { $gte : minCurrentLng } }, { "addressCoordinates.longitude" : { $lte : maxCurrentLng } }] }] } ]})
                .select({ companyName:1, email: 1, addressCoordinates: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1 }).sort({createdAt: -1});
            } else {
                providerList = await Provider.find({ isVerified: true, isApproved: true, $or: [{'physicalAddress.city': { $regex: req.query.cityState, $options: 'i' }}, {'physicalAddress.state': { $regex: req.query.cityState, $options: 'i' }}, {$and: [{ $and: [{ "addressCoordinates.latitude" : { $gte : minCurrentLat } }, { "addressCoordinates.latitude" : { $lte : maxCurrentLat } }] }, { $and: [{ "addressCoordinates.longitude" : { $gte : minCurrentLng } }, { "addressCoordinates.longitude" : { $lte : maxCurrentLng } }] }] }] })
                .select({ companyName:1, email: 1, addressCoordinates: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1 }).sort({createdAt: -1});
            }
            
        } else if(req.query.zip) {
            providerList = await Provider.find({ isVerified: true, isApproved: true, $or: [{'physicalAddress.zip': { $regex: req.query.zip, $options: 'i' }}, {$and: [{ $and: [{ "addressCoordinates.latitude" : { $gte : minCurrentLat } }, { "addressCoordinates.latitude" : { $lte : maxCurrentLat } }] }, { $and: [{ "addressCoordinates.longitude" : { $gte : minCurrentLng } }, { "addressCoordinates.longitude" : { $lte : maxCurrentLng } }] } ]}] })
            .select({ companyName: 1, email: 1, addressCoordinates: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1 }).sort({createdAt: -1});
        } else {
            providerList = await Provider.find({ isVerified: true, isApproved: true }).select({ companyName: 1, email: 1, addressCoordinates: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1 }).sort({createdAt: -1});
        }

        let message = providerList.length == 0 ? 'No provider found in your search area. We also searched 25 mile radius distance around your search location.' : 'Provider list successfully retrieved.';
        
        return HTTP_SUCCESS_RESPONSE(res, { data: providerList }, message );
    }),

    refresh: (req, res) => {
        let token = jwt.sign({ 
            _id: req.provider._id, 
            email: req.provider.email
            },
            config.secret, { expiresIn: '60m' }
        );
        let refreshToken = jwt.sign({ _id: req.provider._id, email: req.provider.email }, config.secret, { expiresIn: '120m' });
        res.setHeader("Authorization", token);
        return HTTP_SUCCESS_RESPONSE(res, { auth: true, token: token, refreshToken: refreshToken }, 'Refresh Request Successfull');
    },

    me: catchAsync(async(req, res) => {
        let id = req.provider._id;
        let provider = await Provider.findById(id).select({ companyName: 1, mainContactFirstName: 1, mainContactLastName: 1, certifiedMedicalExaminarFirstName : 1, certifiedMedicalExaminarLastName: 1, phone: 1, email: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1, nrcmeNo: 1, nrcmeExpirationDate: 1, onlineAppointmentCalendarLink: 1, practitionerType: 1, appointmentType: 1, providerType: 1, isApproved: 1, createdAt: 1, addressCoordinates: 1 });
        if(!provider) throw new HTTP404Error('No provider found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Provider successfully retrieved');
    }),

    userDetails: catchAsync(async(req, res) => {
        let id = req.params.id;
        let provider = await Provider.findById(id).select({ companyName: 1, mainContactFirstName: 1, mainContactLastName: 1, certifiedMedicalExaminarFirstName : 1, certifiedMedicalExaminarLastName: 1, phone: 1, email: 1, physicalAddress: 1, mailingAddress: 1, testPrice: 1, nrcmeNo: 1, nrcmeExpirationDate: 1, onlineAppointmentCalendarLink: 1, practitionerType: 1, appointmentType: 1, providerType: 1, isApproved: 1, createdAt: 1, addressCoordinates: 1 });
        if(!provider) throw new HTTP404Error('No provider found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Successfully provider retrieved');
    }),

    destroy: catchAsync(async(req, res) => {
        let id = req.params.id;
        let provider = await Provider.findOne({ _id: id });
        if(!provider) throw new HTTP404Error('No provider found.');
        await provider.delete();
        return HTTP_SUCCESS_RESPONSE(res, {}, 'Provider successfully deleted');
    }),

    verifyProvider: catchAsync(async(req, res) => {
        let alreadyVerifiedUrl = `${process.env.PROVIDER_PANEL_BASE_URL}/email-verification-exist`;
        let successUrl = `${process.env.PROVIDER_PANEL_BASE_URL}/email-verification-success`;
        let errorUrl = `${process.env.PROVIDER_PANEL_BASE_URL}/email-verification-failed`;

        try {
            const provider = await Provider.findOne({ _id: req.params.id });
            if (!provider) {
                return res.redirect(errorUrl);
            }

            if(provider.isVerified) {
                return res.redirect(alreadyVerifiedUrl);
            }

            const providerByToken = await Provider.findOne({
                token: req.params.token,
            });

            if (!providerByToken) {
                return res.redirect(errorUrl);
            }

            provider.isVerified = true;
            provider.token = null;
            provider.save();
            return res.redirect(successUrl);
        } catch (error) {
            return res.redirect(errorUrl);
        }
    }),

    sendEmailVerificationLink: async (req, res) => {
        try {
            const provider = await Provider.findOne({ email: req.body.email });
            if (!provider) {
            return res.status(400).send({ message: "Provider not exist." });
            }

            provider.token = crypto.randomBytes(32).toString("hex");
            provider.save();

            const template = handlebars.compile(verifyEmailTemplate);
            let message = (template({
                verifyEmailLink: `${process.env.API_BASE_URL}/api/v1/provider/users/verify/${provider._id}/${provider.token}`,
            }));

            let sendEmailData = await sendEmail(provider.email, "Verify Your Email", message);
            if(sendEmailData.status == 200) {
                return res
                .status(200)
                .json({ message: "Successfully send verification email", success: true });
            } else {
                return res
                .status(400)
                .json({ message: sendEmailData.message, success: false });
            }
            
        } catch (error) {
            return res.status(400).send({ message: error });
        }
    },

    forgotPassword: async (req, res) => {
        try {
            const provider = await Provider.findOne({ email: req.body.email });
            if (!provider)
                return res
                .status(400)
                .send({ message: "Provider with given email doesn't exist" });

            provider.token = crypto.randomBytes(32).toString("hex");
            await provider.save();
        
            const template = handlebars.compile(resetPasswordTemplate);
            let message = (template({
                resetPasswordLink: `${process.env.PROVIDER_PANEL_BASE_URL}/password-reset?id=${provider._id}&&token=${provider.token}`,
            }));
        
            sendEmail(provider.email, "Password reset", message);
            return res
            .status(200)
            .json({ message: "Successfully password reset link sent to the email.", success: true });
        } catch (error) {
            return res.status(200).send({ message: error, success: false });
        }
    },
    
    passwordReset: async (req, res) => {
        try {
          const provider = await Provider.findById(req.params.id);
          if (!provider)
            return res.status(400).send({ message: "Provider not exist by this Id." });
    
            const providerByToken = await Provider.findOne({
                token: req.params.token,
            });
            if (!providerByToken)
                return res.status(400).send({ message: "Invalid link or expired" });

            if(req.body.password !== req.body.confirmPassword) {
                return res.status(400).send({ message: "Password and confirm password does not matched." });
            }

            let hashedPassword = bcrypt.hashSync(req.body.password, 8);
            provider.password = hashedPassword;
            provider.token = null;
            await provider.save();
    
            return res
                .status(200)
                .json({ message: "Password Successfully Changed.", success: true });
        } catch (error) {
            return res.status(400).send({ message: error });
        }
    },

    driversOrdersList: catchAsync(async(req, res) => {
        const { page = 1, limit = 5, status = 'all' } = req.query;
        let ordersList = [];
        let count = 0;
        if(status == 'all') {
            ordersList = await Order.find({ provider: req.provider._id })
                .populate({path:'driver', select:['firstName','lastName', 'email', 'phone']})
                .populate({path:'payment', select:['amount','paymentMethod', 'billingAddress', 'uuid']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
            count = await Order.find({ provider: req.provider._id }).count();
        } else {
            ordersList = await Order.find({ provider: req.provider._id, status: status })
                .populate({path:'driver', select:['firstName','lastName', 'email', 'phone']})
                .populate({path:'payment', select:['amount','paymentMethod', 'billingAddress', 'uuid']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
            
            count = await Order.find({ provider: req.provider._id, status: status }).count();
        }

        
        let totalPages = Math.ceil(count / limit);
        return HTTP_SUCCESS_RESPONSE(res, { data: ordersList, totalPages, page }, 'Orders list successfully retrieved by provider.' );
    }),

    driversOrderDetails: catchAsync(async(req, res) => {
        const id = req.params.id;
        let order = await Order.findById(id)
            .populate({path:'driver', select:['firstName','lastName', 'email', 'phone']})
            .populate({path:'payment', select:['amount','paymentMethod', 'billingAddress', 'uuid']})
            .sort({createdAt: -1})
            .exec();
        if(!order) throw new HTTP404Error('No order found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: order }, 'Order details successfully retrieved.' );
    }),

    driversOrderUpdate: catchAsync(async(req, res) => {
        const { id, dateOfExam, certificateExpirationDate, isOrderCompleted, isOrderDisqualified, isOrderDeterminationPending, disqualifiedReason, determinationPendingReason } = req.body;
        let order = await Order.findById(id);
        if(!order) throw new HTTP404Error('No order found.');

        order.dateOfExam = dateOfExam;
        if(isOrderCompleted == true) {
            order.certificateExpirationDate = certificateExpirationDate;
            order.status = "completed";
        } else if(isOrderDisqualified == true) {
            order.status = "disqualified";
        } else if(isOrderDeterminationPending == true) {
            order.status = "determination-pending";
        }
        order.isOrderCompleted = isOrderCompleted;
        order.isOrderDisqualified = isOrderDisqualified;
        order.isOrderDeterminationPending = isOrderDeterminationPending;
        order.disqualifiedReason = disqualifiedReason;
        order.determinationPendingReason = determinationPendingReason;
        order.save();
        return HTTP_SUCCESS_RESPONSE(res, { data: order }, 'Order data successfully updated.' );
    }),

    countDriversOrders: catchAsync(async(req, res) => {
        let countTotalOrders = await Order.find({ provider: req.provider._id }).count();
        let countProcessingOrders = await Order.find({ provider: req.provider._id, status: 'processing' }).count();
        let completedOrders = await Order.find({ provider: req.provider._id, status: 'completed' }).populate({path:'payment', select:['amount']}).exec();
        let countDeterminationPendingOrders = await Order.find({ provider: req.provider._id, status: 'determination-pending' }).count();
        let countDisqualifiedOrders = await Order.find({ provider: req.provider._id, status: 'disqualified'}).count();

        let countCompletedOrders = completedOrders.length;
        let totalEarning = 0;
        await completedOrders.forEach((item) => {
            totalEarning += Number(item.payment.amount) - 15;
        });

        return HTTP_SUCCESS_RESPONSE(res, { data: { countTotalOrders, countProcessingOrders, countCompletedOrders, countDeterminationPendingOrders, countDisqualifiedOrders, totalEarning } }, 'Orders list successfully retrieved by provider.' );
    }),

    countTotalProviders: catchAsync(async(req, res) => {
        let countProviders = await Provider.countDocuments();
        return HTTP_SUCCESS_RESPONSE(res, { data: countProviders }, 'Successfully count providers.' );
    }),

    exportExcel: async (req, res) => {
        try {
            let providers = await Provider.find({}).select({ companyName: 1,  phone: 1, email: 1, physicalAddress: 1, nrcmeNo: 1 }).sort({createdAt: -1});
            providers = providers.map((item) => {
                let data = {
                    companyName: item.companyName,
                    phone: item.phone,
                    email: item.email,
                    physicalAddress: item.physicalAddress.street + ', ' + item.physicalAddress.city + ', ' +  item.physicalAddress.state + ', ' + item.physicalAddress.zip,
                    nrcmeNo: item.nrcmeNo,
                }

                return data;
            })

            const worksheet = XLSX.utils.json_to_sheet(providers);
            const workbook = XLSX.utils.book_new();
            XLSX.utils.book_append_sheet(workbook, worksheet, "Providers");
            XLSX.utils.sheet_add_aoa(worksheet, [["Company Name", "Phone", "Email", "Physical Address", "Nrcme No."]], { origin: "A1" });
            XLSX.writeFile(workbook, "public/export/providers.xlsx");
            return res.status(200).json({ path: `${process.env.API_BASE_URL}/export/providers.xlsx`, message: "Successfully Export Excel" });
        } catch(error) {
            return res.status(400).json({ message: error });
        }
    },

    // Admin Interaction
    approveProviderByAdmin: catchAsync(async(req, res) => {
        let id = req.params.id;
        let provider = await Provider.findOne({ _id: id });
        if(!provider) throw new HTTP404Error('No provider found.');
        provider.isApproved = true;
        provider.save();
        return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Provider successfully approved from admin.');
    }),

    verifyProviderEmailByAdmin: catchAsync(async(req, res) => {
        let id = req.params.id;
        let provider = await Provider.findOne({ _id: id });
        if(!provider) throw new HTTP404Error('No provider found.');
        provider.isVerified = true;
        provider.save();
        return HTTP_SUCCESS_RESPONSE(res, { data: provider }, 'Provider email successfully verified from admin.');
    }),
}
