const nanoidAsync = require('nanoid/async');
const bcrypt = require('bcrypt');
const crypto = require("crypto");
const mongoose = require('mongoose');
const sendEmail = require("../utils/sendEmail");
const { HTTP_SUCCESS_RESPONSE } = require("../utils/success");
const { HTTP400Error, HTTP404Error, catchAsync } = require("../utils/errors");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);
const QRCode = require("qrcode");
const XLSX = require("xlsx");

const handlebars = require("handlebars");
const fs = require("fs");
const path = require("path");
const orderEmailTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/order.handlebars"), "utf-8"); 
const orderProviderEmailTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/orderProviderEmailTemplate.handlebars"), "utf-8"); 
const resetPasswordTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/resetPassword.handlebars"), "utf-8"); 

// Import model
const Provider = require('../models/providers.js');
const Driver = require('../models/drivers.js');
const Payment = require('../models/payments.js');
const Order = require('../models/orders.js');

const formatDate = (value) => {
    let date = new Date(value);
    let year = date.getFullYear();
    let month = date.getMonth() + 1;
    let dt = date.getDate();
  
    if (dt < 10) {
      dt = "0" + dt;
    }
    if (month < 10) {
      month = "0" + month;
    }
    return month + "-" + dt + "-" + year;
};

module.exports = {
    create: catchAsync(async (req, res) => {
        const paymentData = { provider: providerId, billingAddress, amount, paymentMethod } = req.body;
        let provider = await Provider.findById(providerId);
        if(!provider) throw new HTTP404Error('Please select a correct provider.');
        let payment = new Payment(paymentData);
        payment.driver = req.driver._id;
        payment.uuid = await nanoidAsync.nanoid();
        return payment.save()
        .then(async function(payment) {
            await payment.setCreatedBy(req.params._id); 
            await payment.setUpdatedBy(req.params._id);
            await payment.save();
            try {
                let order = new Order();
                order.provider = providerId;
                order.driver = req.driver._id;
                order.payment = payment._id;
                order.status = 'pending';
                order.uuid = await nanoidAsync.nanoid();
                await order.setCreatedBy(req.params._id); 
                await order.setUpdatedBy(req.params._id);
                await order.save();
                return HTTP_SUCCESS_RESPONSE(res, { data: payment, order: order }, 'Payment and order Successfully Created.' ); 
            } catch(err) {
                throw new HTTP404Error('Something went wrong in creating order.');
            }
        }).catch(err => {
            throw new HTTP400Error(err.message);
        });    
    }),

    list: catchAsync(async(req, res) => {
        let paymentList = await Payment.find({}).select({ billingAddress: 1, amount: 1, paymentMethod: 1, uuid: 1 })
                        .populate({path:'driver', select:['firstName','lastName']}).populate({path:'provider', select:['firstName','lastName']})
                        .sort({created_at: -1});
        return HTTP_SUCCESS_RESPONSE(res, { data: paymentList }, 'Payment list successfully retrieved.' );
    }),

    paymentDetails: catchAsync(async(req, res) => {
        let id = req.params.id;
        let payment = await Payment.findOne({ _id: id }).populate({path:'driver', select:['firstName','lastName']}).populate({path:'provider', select:['firstName','lastName']});
        if(!payment) throw new HTTP404Error('No payment found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: payment }, 'Payment successfully retrieved');
    }),
}

// For Orders Endpoint
module.exports = {
    ordersList: catchAsync(async(req, res) => {
        const { page = 1, limit = 5, status = 'all', dateSorting = 'all' } = req.query;
        let ordersList = [];
        let count = 0;

        let compareDate = new Date();
        function getFirstDayOfWeek(d) {
            const date = new Date(d);
            const day = date.getDay();
            const diff = date.getDate() - day;
          
            return new Date(date.setDate(diff));
        }

        function getFirstDayOfMonth(year, month) {
            return new Date(year, month, 1);
        }

        if(dateSorting == 'weekly') {
            compareDate = getFirstDayOfWeek(new Date());
        } else if (dateSorting == 'monthly') {
            const date = new Date();
            compareDate = getFirstDayOfMonth(
                date.getFullYear(),
                date.getMonth(),
            );
        }

        if(status == 'all') {
            if(dateSorting == 'all') {
                ordersList = await Order.find({}).select({ status: 1, expirationDate: 1, uuid: 1, createdAt: 1 })
                .populate({path:'driver', select:['firstName','lastName','email', 'phone']})
                .populate({path:'provider', select:['companyName', 'email', 'phone']})
                .populate({path:'payment', select:['amount','orderMethod', 'billingAddress', 'uuid']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
                count = await Order.countDocuments();
            } else {
                ordersList = await Order.find({ createdAt: { $gte: compareDate }}).select({ status: 1, expirationDate: 1, uuid: 1, createdAt: 1 })
                .populate({path:'driver', select:['firstName','lastName','email', 'phone']})
                .populate({path:'provider', select:['companyName', 'email', 'phone']})
                .populate({path:'payment', select:['amount','orderMethod', 'billingAddress', 'uuid']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
                count = await Order.find({ createdAt: { $gte: compareDate }}).countDocuments();
            }

        } else {
            if(dateSorting == 'all') {
                ordersList = await Order.find({ status: status }).select({ status: 1, expirationDate: 1, uuid: 1, createdAt: 1 })
                .populate({path:'driver', select:['firstName','lastName','email', 'phone']})
                .populate({path:'provider', select:['companyName', 'email', 'phone']})
                .populate({path:'payment', select:['amount','orderMethod', 'billingAddress', 'uuid']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
                count = await Order.find({ status: status }).countDocuments();
            } else {
                ordersList = await Order.find({ status: status, createdAt: { $gte: compareDate } }).select({ status: 1, expirationDate: 1, uuid: 1, createdAt: 1 })
                .populate({path:'driver', select:['firstName','lastName','email', 'phone']})
                .populate({path:'provider', select:['companyName', 'email', 'phone']})
                .populate({path:'payment', select:['amount','orderMethod', 'billingAddress', 'uuid']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
                count = await Order.find({ status: status, createdAt: { $gte: compareDate } }).countDocuments();
            }
        }

        let totalPages = Math.ceil(count / limit);
        return HTTP_SUCCESS_RESPONSE(res, { data: ordersList, totalPages, page },'Orders list successfully retrieved.' );
    }),

    ordersSalesReport: catchAsync(async(req, res) => {
        const { page = 1, limit = 5, dateSorting = 'all' } = req.query;
        let ordersList = [];
        let countDocuments = [];

        let compareDate = new Date();
        function getFirstDayOfWeek(d) {
            const date = new Date(d);
            const day = date.getDay();
            const diff = date.getDate() - day;
          
            return new Date(date.setDate(diff));
        }

        function getFirstDayOfMonth(year, month) {
            return new Date(year, month, 1);
        }

        if(dateSorting == 'weekly') {
            compareDate = getFirstDayOfWeek(new Date());
        } else if (dateSorting == 'monthly') {
            const date = new Date();
            compareDate = getFirstDayOfMonth(
                date.getFullYear(),
                date.getMonth(),
            );
        }

        if(dateSorting == 'all') {
            ordersList = await Order.aggregate([
                {
                    $match: {
                        $and:[{"status" : { $ne: 'processing'} }]
                    }
                },
    
                {
                    $lookup:
                       {
                          from: "providers",
                          localField: "provider",
                          foreignField: "_id",
                          as: "provider_info"
                      }
                },
    
                {   $unwind:"$provider_info" },
    
                {
                    $lookup:
                        {
                            from: "payments",
                            localField: "payment",
                            foreignField: "_id",
                            pipeline : [
                                { $project : { _id:1, amount:1 } }
                            ],
                            as: "payment_info"
                        }
                },
    
                {
                    "$group": {
                        "_id": "$provider",
                        "provider": {"$first": "$provider_info" },
                        "payment": { "$push": "$payment_info" },
                    }
                },
    
                {
                    $limit : limit * 1
                },
    
                {
                    $skip : (page - 1) * limit
                }
            ]);
    
            countDocuments =  await Order.aggregate([
                {
                    $match: {
                        $and:[{"status" : { $ne: 'processing'} }]
                    }
                },

                {
                    $lookup:
                       {
                          from: "providers",
                          localField: "provider",
                          foreignField: "_id",
                          as: "provider_info"
                      }
                },
    
                {   $unwind:"$provider_info" },
    
                {
                    "$group": {
                        "_id": "$provider",
                    }
                },
            ]);
        } else {
            ordersList = await Order.aggregate([
                {
                    $match: {
                        $and:[{"status" : { $ne: 'processing'} }, {"createdAt" : { $gte: compareDate }}]
                    }
                },
    
                {
                    $lookup:
                       {
                          from: "providers",
                          localField: "provider",
                          foreignField: "_id",
                          as: "provider_info"
                      }
                },
    
                {   $unwind:"$provider_info" },
    
                {
                    $lookup:
                        {
                            from: "payments",
                            localField: "payment",
                            foreignField: "_id",
                            pipeline : [
                                { $project : { _id:1, amount:1 } }
                            ],
                            as: "payment_info"
                        }
                },
    
                {
                    "$group": {
                        "_id": "$provider",
                        "provider": {"$first": "$provider_info" },
                        "payment": { "$push": "$payment_info" },
                    }
                },
    
                {
                    $limit : limit * 1
                },
    
                {
                    $skip : (page - 1) * limit
                }
            ]);
    
            countDocuments =  await Order.aggregate([
                {
                    $match: {
                        $and:[{"status" : { $ne: 'processing'}}, {"createdAt" : { $gte: compareDate } }]
                    }
                },

                {
                    $lookup:
                       {
                          from: "providers",
                          localField: "provider",
                          foreignField: "_id",
                          as: "provider_info"
                      }
                },
    
                {   $unwind:"$provider_info" },
    
                {
                    "$group": {
                        "_id": "$provider",
                    }
                },
            ]);
        }

        console.log(countDocuments.length, 'counte');
        let totalPages = Math.ceil(countDocuments.length / limit);
        return HTTP_SUCCESS_RESPONSE(res, { data: ordersList, page, totalPages }, 'Orders list successfully retrieved 1111.' );
    }),

    providerOrdersSalesReport: catchAsync(async(req, res) => {
        const providerId = req.params.id;
        const { page = 1, limit = 5, sortingMonth = '', sortingYear = '' } = req.query;
        providerOrdersList = [];
        countDocuments = 0;

        let startOfTheDate = new Date();
        let endOfTheDate = new Date();

        function getStartOfTheDate(year, month) {
            return new Date(year, month, 1);
        }

        function getEndOfTheDate(year, month) {
            return new Date(year, month + 1, 0);
        }



        if (sortingMonth && sortingYear) {

            console.log(sortingMonth, sortingYear, 'lloo');
            startOfTheDate = getStartOfTheDate(
                Number(sortingYear),
                Number(sortingMonth),
            );

            endOfTheDate = getEndOfTheDate(
                Number(sortingYear),
                Number(sortingMonth),
            );

            providerOrdersList = await Order.find({ provider: mongoose.Types.ObjectId(providerId), $and: [{"status" : { $ne: 'processing'}}, {'createdAt': { $gte: startOfTheDate }}, {'createdAt': { $lte: endOfTheDate }}]})
                .populate({path:'driver', select:['firstName','lastName', 'phone', 'email']})
                .populate({path:'payment', select:['amount']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
            countDocuments = await Order.find({ provider: mongoose.Types.ObjectId(providerId), $and: [{"status" : { $ne: 'processing'}}, {'createdAt': { $gte: startOfTheDate }}, {'createdAt': { $lte: endOfTheDate }}]}).countDocuments();

        } else {
            providerOrdersList = await Order.find({ provider: mongoose.Types.ObjectId(providerId), $and: [{"status" : { $ne: 'processing'}}] })
                .populate({path:'driver', select:['firstName','lastName', 'phone', 'email']})
                .populate({path:'payment', select:['amount']})
                .sort({createdAt: -1})
                .limit(limit * 1)
                .skip((page - 1) * limit)
                .exec();
            countDocuments = await Order.find({ provider: mongoose.Types.ObjectId(providerId), $and: [{"status" : { $ne: 'processing'}}] }).countDocuments();
        }

        
        let totalPages = Math.ceil(countDocuments / limit);
        return HTTP_SUCCESS_RESPONSE(res, { data: providerOrdersList, page, totalPages }, 'Provider orders list successfully retrieved.' );
    }),

    providersExportExcel: catchAsync(async(req, res) => {
            try {
                const providerId = req.params.id;
                const { sortingMonth = '', sortingYear = '' } = req.query;
                providerOrdersList = [];
                countDocuments = 0;

                let startOfTheDate = new Date();
                let endOfTheDate = new Date();

                function getStartOfTheDate(year, month) {
                    return new Date(year, month, 1);
                }

                function getEndOfTheDate(year, month) {
                    return new Date(year, month + 1, 0);
                }



                if (sortingMonth && sortingYear) {

                    console.log(sortingMonth, sortingYear, 'lloo');
                    startOfTheDate = getStartOfTheDate(
                        Number(sortingYear),
                        Number(sortingMonth),
                    );

                    endOfTheDate = getEndOfTheDate(
                        Number(sortingYear),
                        Number(sortingMonth),
                    );

                    providerOrdersList = await Order.find({ provider: mongoose.Types.ObjectId(providerId), $and: [{"status" : { $ne: 'processing'}}, {'createdAt': { $gte: startOfTheDate }}, {'createdAt': { $lte: endOfTheDate }}]})
                        .populate({path:'provider', select:['companyName','email']})
                        .populate({path:'driver', select:['firstName','lastName', 'phone', 'email']})
                        .populate({path:'payment', select:['amount']})
                        .sort({createdAt: -1});

                } else {
                    providerOrdersList = await Order.find({ provider: mongoose.Types.ObjectId(providerId), $and: [{"status" : { $ne: 'processing'}}]})
                    .populate({path:'provider', select:['companyName','email']})
                    .populate({path:'driver', select:['firstName','lastName', 'phone', 'email']})
                    .populate({path:'payment', select:['amount']})
                    .sort({createdAt: -1});
                }

                providerOrdersList = providerOrdersList.map((item) => {
                    let data = {
                        providerCompanyName: item.provider?item.provider.companyName: "N/A",
                        providerEmail: item.provider?item.provider.email: "N/A",
                        driverName: item.driver?item.driver.firstName + " " + item.driver.lastName : "N/A",
                        driverEmail: item.driver?item.driver.email: "N/A",
                        totalAmount: item.payment?item.payment.amount: "N/A",
                        createdAt: formatDate(item.createdAt)
                    }

                    return data;
                })

                const worksheet = XLSX.utils.json_to_sheet(providerOrdersList);
                const workbook = XLSX.utils.book_new();
                XLSX.utils.book_append_sheet(workbook, worksheet, "Provider Orders");
                XLSX.utils.sheet_add_aoa(worksheet, [["Provider Company Name", "Prover Email", "Driver Name", "Driver Email", "Total Amount", "Created At"]], { origin: "A1" });
                XLSX.writeFile(workbook, "public/export/provider_orders.xlsx");

                return res.status(200).json({ path: `${process.env.API_BASE_URL}/export/provider_orders.xlsx`, message: "Successfully Provider Orders export" });
            } catch(error) {
                return res.status(400).json({ message: error });
            }
        }),

        orderDetails: catchAsync(async(req, res) => {
            let id = req.params.id;
            let order = await Order.findOne({ _id: id }).populate({path:'driver', select:['firstName','lastName', 'phone', 'email']})
                        .populate({path:'provider', select:['companyName', 'phone', 'email']}).populate({path:'payment', select:['amount','orderMethod', 'billingAddress', 'uuid']});
            if(!order) throw new HTTP404Error('No order found.');
            return HTTP_SUCCESS_RESPONSE(res, { data: order }, 'Order successfully retrieved');
        }),

        countOrdersFromAdmin: catchAsync(async(req, res) => {
            let totalOrders = await Order.find({}).populate({path:'payment', select:['amount']}).exec();;
            let countProcessingOrders = await Order.find({ status: 'processing' }).count();
            let countCompletedOrders = await Order.find({ status: 'completed' }).count();
            let countDeterminationPendingOrders = await Order.find({ status: 'determination-pending' }).count();
            let countDisqualifiedOrders = await Order.find({ status: 'disqualified'}).count();

            let countTotalOrders = totalOrders.length;
            let totalDepositAmount = 0;
            await totalOrders.forEach((item) => {
                totalDepositAmount += Number(item.payment.amount);
            });

            return HTTP_SUCCESS_RESPONSE(res, { data: { countTotalOrders, totalDepositAmount, countProcessingOrders, countCompletedOrders, countDeterminationPendingOrders, countDisqualifiedOrders } }, 'Successfully count orders.' );
    }),

    createPaymentOrder: async (req, res) => {
        const { providerId, driverInfo, amount, paymentMethod, paymentInfo } = req.body;
        let provider = await Provider.findById(providerId);
        if(!provider) throw new HTTP404Error('Please select a correct provider.');

        let driverId = "";
        let driverEmail = "";
        let driver = {};
        driverId = req.driver ? req.driver._id : "";
        driverEmail = req.driver ? req.driver.email : "";
        if(req.driver) {
            driver = await Driver.findById(req.driver._id);
        }
        if(!req.driver) {
            let hashedPassword = bcrypt.hashSync('12345678', 8)
            driver = new Driver();
            driver.firstName = driverInfo.firstName;
            driver.lastName = driverInfo.lastName;
            driver.email = driverInfo.email;
            driver.phone = driverInfo.phone;
            driver.physicalAddress = driverInfo.address;
            driver.mailingAddress = driverInfo.address;
            driver.password = hashedPassword;
            driver.isVerified = true;
            driver.token = crypto.randomBytes(32).toString("hex");
            await driver.save()
            .then((driver) => {
                driverId = driver._id;
                driverEmail = driver.email;

                const template = handlebars.compile(resetPasswordTemplate);
                let message = (template({
                    resetPasswordLink: `${process.env.DRIVER_PANEL_BASE_URL}/password-reset?id=${driver._id}&&token=${driver.token}`,
                }));

                sendEmail(driver.email, "Reset password for login dotphys driver panel.", message);
            })
            .catch((error) => {
                console.log(error);
            })
        }

        console.log(driverId, "dsfdsfsdf");

        let payment = new Payment();
        payment.provider = providerId;
        payment.driver = driverId;
        payment.billingAddress = driverInfo.address;
        payment.amount = amount;
        payment.paymentMethod = paymentMethod;
        payment.stripeId = paymentInfo.id;
        payment.uuid = await nanoidAsync.nanoid();
        return payment.save()
        .then(async function(payment) {
            await payment.setCreatedBy(driverId); 
            await payment.setUpdatedBy(driverId);
            await payment.save();
            try {
                let order = new Order();
                order.provider = providerId;
                order.driver = driverId;
                order.payment = payment._id;
                order.status = 'processing';
                order.uuid = "ORDER_" + await nanoidAsync.nanoid();
                order.token = crypto.randomBytes(32).toString("hex");
                await order.setCreatedBy(driverId); 
                await order.setUpdatedBy(driverId);
                await order.save();

                let qrImg = 'QR_'+(Math.random() * 10000000);

                let qrMessage = `Name: ${driver.firstName + ' ' + driver.lastName } \n`;
                qrMessage += `Email: ${driver.email} \n`;
                qrMessage += `Phone: ${driver.phone} \n`;
                qrMessage += `Address: ${driver.physicalAddress.street + ', ' + driver.physicalAddress.city + ', ' + 
                driver.physicalAddress.state + ', ' + driver.physicalAddress.zip + '.'} \n`;
                qrMessage += `Redeemption Link: <a href="${process.env.PROVIDER_PANEL_BASE_URL}/redeem-order/${order._id}">Click Here</a>`;
                
                let qrImgPath = path.join(__dirname, `../../public/qr_images/${order.uuid}.png`);
                let qrImgSrc = `${process.env.API_BASE_URL}/qr_images/${order.uuid}.png`;

                console.log(qrImgPath, qrImgSrc);

                QRCode.toFile(qrImgPath, qrMessage , {
                    color: {
                      dark: '#f00', 
                      light: '#0000'
                    }
                }, function (err) {
                    if (err) {
                        console.log(err);
                    } else {
                        console.log('Qr code successfully generates');
                    }
                });

                let providerData = {
                    companyName: provider.companyName,
                    address: provider.physicalAddress,
                    phone: provider.phone,
                    email: provider.email,
                    onlineAppointmentCalendarLink: provider.onlineAppointmentCalendarLink,
                    appointmentType: provider.appointmentType
                };

                let driverDetails = {
                    name: driver.firstName + " " + driver.lastName,
                    phone: driver.phone,
                    email: driver.email
                }

                const template = handlebars.compile(orderEmailTemplate);
                let message = (template({
                    amount: amount,
                    driverDetails: driverDetails,
                    order_uuid: order.uuid,
                    providerDetails: providerData,
                    qrImgSrc: qrImgSrc
                }));

                sendEmail(driverEmail, "$65 DOT Physicals order create successfully", message);

                const providerTemplate = handlebars.compile(orderProviderEmailTemplate);
                let providerEmailMessage = (providerTemplate({
                    driverDetails: driverDetails,
                    redeemptionLink: `${process.env.PROVIDER_PANEL_BASE_URL}/redeem-order/${order._id}`
                }));

                sendEmail(provider.email, "$65 DOT Physicals Driver subscribed your services", providerEmailMessage);


                // let messageProvider = "<h4>Congratulations! One driver is subscribed your services.</h4>";
                // messageProvider += `<b>Driver email is: ${ driverEmail } </b>`;
                // sendEmail(provider.email, "$65 DOT Physicals Driver subscribed your services.", messageProvider);

                return HTTP_SUCCESS_RESPONSE(res, { data: payment, order: order }, 'Payment and order successfully created.' ); 
            } catch(err) {
                console.log(err);
                throw new HTTP404Error('Something went wrong in creating order.');
            }
        }).catch(err => {
            console.log(err);
            throw new HTTP400Error(err.message);
        });   


    },

    stripePaymentInitialization: async (req, res) => {
        const paymentIntent = await stripe.paymentIntents.create({
            amount: "6500",
            currency: "usd"
        });
        return res.send({clientSecret: paymentIntent.client_secret });
    },
}