const nanoidAsync = require('nanoid/async');
const { HTTP_SUCCESS_RESPONSE } = require("../utils/success");
const { HTTP400Error, HTTP404Error, catchAsync } = require("../utils/errors");

// Import model
const Order = require('../models/orders.js');
const Appointment = require('../models/appointments.js');

module.exports = {
    create: catchAsync(async (req, res) => {
        const { appointmentDateTime } = req.body;
        let orderId = req.params.orderId;
        let order = await Order.findById(orderId);
        if(!order) throw new HTTP404Error('Please select a correct order for appointment.');
        let appointment = new Appointment();
        appointment.provider = order.provider;
        appointment.driver = order.driver;
        appointment.payment = order.payment;
        appointment.order = order._id;
        appointment.appointmentDateTime = appointmentDateTime;
        appointment.uuid = await nanoidAsync.nanoid();
        return appointment.save()
        .then(async function(appointment) {
            await appointment.setCreatedBy(order.driver); 
            await appointment.setUpdatedBy(order.driver);
            await appointment.save();
            return HTTP_SUCCESS_RESPONSE(res, { data: appointment }, 'Appointment Successfully Created.' ); 
        }).catch(err => {
            throw new HTTP400Error(err.message);
        });    
    }),

    update: catchAsync(async (req, res) => {
        const { appointmentDateTime } = req.body;
        let id = req.params.id;
        let appointment = await Appointment.findById(id);
        if(!appointment) throw new HTTP404Error('No appointment found.');
        appointment = appointment.toJSON();
        appointment.appointmentDateTime = appointmentDateTime;

        try {
            let updateAppointment = await Appointment(appointment);
            await updateAppointment.setUpdatedBy(req.driver._id); 
            await updateAppointment.setUpdatedAt();
            updateAppointment = await Appointment(updateAppointment); 
            return HTTP_SUCCESS_RESPONSE(res, { data: updateAppointment }, 'Appointment successfully updated' );
        } catch(err) {
            throw new HTTP400Error(err.message);
        }
    }),

    list: catchAsync(async(req, res) => {
        let appointmentList = await Appointment.find({}).select({ appointmentDateTime: 1, appointmentType: 1, uuid: 1 })
                        .populate({path:'driver', select:['firstName','lastName']}).populate({path:'provider', select:['firstName','lastName']})
                        .populate({path:'payment', select:['amount','billingAddress', 'uuid']}).populate({path:'order', select:['status','expirationDate', 'uuid']})
                        .sort({created_at: -1});
        return HTTP_SUCCESS_RESPONSE(res, { data: appointmentList }, 'Appointment list successfully retrieved.' );
    }),

    appointmentDetails: catchAsync(async(req, res) => {
        let id = req.params.id;
        let appointment = await Appointment.findOne({ _id: id })
                        .populate({path:'driver', select:['firstName','lastName']}).populate({path:'provider', select:['firstName','lastName']})
                        .populate({path:'payment', select:['amount','billingAddress', 'uuid']}).populate({path:'order', select:['status','expirationDate', 'uuid']})
        if(!appointment) throw new HTTP404Error('No appointment found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: appointment }, 'Appointment successfully retrieved');
    }),

    destroy: catchAsync(async(req, res) => {
        let id = req.params.id;
        let appointment = await Appointment.findOne({ _id: id });
        if(!appointment) throw new HTTP404Error('No appointment found.');
        await appointment.delete();
        return HTTP_SUCCESS_RESPONSE(res, {}, 'Appointment successfully deleted');
    })
}
