const express = require('express');
const router = express.Router();
const { param, body } = require("express-validator");
const { createProviderUserValidation, updateProviderUserValidation, editPasswordProviderUserValidation, testPriceProviderUserValidation, loginProviderUserValidation, passwordResetValidation } = require("../../utils/ApiValidation/provider_users");
const { handleValidationErrors } = require("../../utils/errors");
const providerUsersAction = require('../actions/provider_users.action');
const providerUserMiddleware = require('../middlewares/provider_users.middleware.js');

router
	.post('/',
		createProviderUserValidation,
		handleValidationErrors,
		providerUsersAction.create
	)
	.get(
		'/', 
		providerUserMiddleware.checkToken, 
		providerUsersAction.list
	)
	.post(
		'/login',
		loginProviderUserValidation,
		handleValidationErrors,
		providerUsersAction.login
	)
	.get(
		'/refresh', 
		providerUserMiddleware.checkToken, 
		providerUsersAction.refresh
	)

	.post(
		'/edit-password',
		providerUserMiddleware.checkToken,
		editPasswordProviderUserValidation,
		handleValidationErrors,
		providerUsersAction.updatePassword
	)

	.post(
		'/update-test-price/:id',
		providerUserMiddleware.checkToken,
		testPriceProviderUserValidation,
		handleValidationErrors,
		providerUsersAction.updatePrice
	)

	.get('/save-coordinates',
		providerUsersAction.saveCoordinates
	)

	.get(
		'/me', 
		providerUserMiddleware.checkToken, 
		providerUsersAction.me
	)

	// Provider list from driver panel

	.get(
		'/providers-list', 
		providerUsersAction.providerListDriverPanel
	)

	// Verify User
	.get(
		"/verify/:id/:token",
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors, 
		providerUsersAction.verifyProvider
	)

	//Send verificatoin email
	.post(
		"/send-email-verification-link",
		body("email")
		  .exists()
		  .withMessage("Email is required.")
		  .isEmail()
		  .withMessage("Email field must be valid."),
		handleValidationErrors,
		providerUsersAction.sendEmailVerificationLink
	)

	.post(
		"/forgot-password",
		body("email")
		  .exists()
		  .withMessage("Email is required.")
		  .isEmail()
		  .withMessage("Email field must be valid."),
		handleValidationErrors,
		providerUsersAction.forgotPassword
	)

	.post(
		"/password-reset/:id/:token",
		passwordResetValidation,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		providerUsersAction.passwordReset
	)

	.get(
		"/drivers-orders-list",
		providerUserMiddleware.checkToken, 
		providerUsersAction.driversOrdersList
	)

	.get(
		"/drivers-order-details/:id",
		providerUserMiddleware.checkToken, 
		providerUsersAction.driversOrderDetails
	)

	.post(
		"/order-update",
		providerUserMiddleware.checkToken, 
		providerUsersAction.driversOrderUpdate
	)

	.get(
		"/count-drivers-orders",
		providerUserMiddleware.checkToken, 
		providerUsersAction.countDriversOrders
	)

	.get(
		'/:id', 
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,  
		providerUsersAction.userDetails
	)
	.put(
		'/:id',
		providerUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		updateProviderUserValidation,
		handleValidationErrors, 
		providerUsersAction.update
	)
	.delete(
		'/:id', 
		providerUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors, 
		providerUsersAction.destroy
	)
module.exports = router