const express = require('express');
const router = express.Router();
const { param, body } = require("express-validator");
const { createDriverUserValidation, updateDriverUserValidation, loginDriverUserValidation, passwordResetValidation } = require("../../utils/ApiValidation/driver_users");
const { handleValidationErrors } = require("../../utils/errors");
const driverUsersAction = require('../actions/driver_users.action');
const driverUserMiddleware = require('../middlewares/driver_users.middleware.js');

router
	.post('/',
		createDriverUserValidation,
		handleValidationErrors,
		driverUsersAction.create
	)
	.get(
		'/', 
		driverUserMiddleware.checkToken, 
		driverUsersAction.list
	)
	.post(
		'/login',
		loginDriverUserValidation,
		handleValidationErrors,
		driverUsersAction.login
	)
	.post(
		'/google-login',
		driverUsersAction.googleLogin
	)
	.post(
		'/facebook-login',
		driverUsersAction.facebookLogin
	)
	.get(
		'/refresh', 
		driverUserMiddleware.checkToken, 
		driverUsersAction.refresh
	)

	.get(
		'/me', 
		driverUserMiddleware.checkToken, 
		driverUsersAction.me
	)

	// Verify User
	.get(
		"/verify/:id/:token",
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors, 
		driverUsersAction.verifyDriver
	)

	//Send verificatoin email
	.post(
		"/send-email-verification-link",
		body("email")
			.exists()
			.withMessage("Email is required.")
			.isEmail()
			.withMessage("Email field must be valid."),
		handleValidationErrors,
		driverUsersAction.sendEmailVerificationLink
	)

	.post(
		"/forgot-password",
		body("email")
			.exists()
			.withMessage("Email is required.")
			.isEmail()
			.withMessage("Email field must be valid."),
		handleValidationErrors,
		driverUsersAction.forgotPassword
	)

	.post(
		"/password-reset/:id/:token",
		passwordResetValidation,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		driverUsersAction.passwordReset
	)

	.get(
		'/:id', 
		driverUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,  
		driverUsersAction.userDetails
	)
	.put(
		'/:id',
		driverUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		updateDriverUserValidation,
		handleValidationErrors, 
		driverUsersAction.update
	)
	.delete(
		'/:id', 
		driverUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors, 
		driverUsersAction.destroy
	)


module.exports = router